<?php
// Function to fetch and pass MFA summary data to JavaScript
function get_mfa_summary_data() {
    global $wpdb;

    $total_users_count = count(get_users());

    $enabled_users = (int)$wpdb->get_var("
        SELECT COUNT(DISTINCT user_email) 
        FROM {$wpdb->prefix}bca_api_entries 
        WHERE qr_status = 'scanned'
    ");

    $not_enabled_users = max(0, $total_users_count - $enabled_users);

    $mfa_logins = (int)$wpdb->get_var("
        SELECT COUNT(*) 
        FROM {$wpdb->prefix}bca_api_entries 
        WHERE auth_type IS NOT NULL
    ");

    $success_auth_types = [
        'Push Notification',
        'TOTP',
        'Security Question',
        'OTP via Email'
    ];
    $success_auth_types_str = "'" . implode("','", array_map('esc_sql', $success_auth_types)) . "'";

    $successful_users = (int)$wpdb->get_var("
        SELECT COUNT(DISTINCT user_email)
        FROM {$wpdb->prefix}bca_api_entries
        WHERE qr_status = 'scanned' AND auth_type IN ($success_auth_types_str)
    ");

    $attempted_users = $enabled_users; // same query logic

    $mfa_success_rate = $attempted_users > 0 
        ? round(($successful_users / $attempted_users) * 100, 2) 
        : 0;

    $locked_accounts = (int)$wpdb->get_var("
        SELECT COUNT(DISTINCT u.ID)
        FROM {$wpdb->users} u
        LEFT JOIN {$wpdb->prefix}usermeta um ON u.ID = um.user_id
        WHERE um.meta_key = 'account_status' AND um.meta_value = 'inactive'
    ");

    $unlocked_accounts = max(0, $total_users_count - $locked_accounts);

    function getMFAData($authType) {
        $map = [
            'Push Notification' => ['label' => 'Push', 'color' => '#16478c'],
            'TOTP'              => ['label' => 'TOTP', 'color' => '#03a7dd'],
            'Security Question' => ['label' => 'KBA',  'color' => '#eb8c22'],
            'OTP via Email'     => ['label' => 'Email', 'color' => '#9b59b6'],
        ];
        return $map[$authType] ?? ['label' => ucfirst($authType), 'color' => '#2ecc71'];
    }

    $mfa_methods = $wpdb->get_results("
        SELECT auth_type, COUNT(*) AS method_count
        FROM {$wpdb->prefix}bca_api_entries
        WHERE auth_type IS NOT NULL
        GROUP BY auth_type
    ");

    $mfa_method_labels = [];
    $mfa_method_counts = [];
    $mfa_method_colors = [];

    foreach ($mfa_methods as $method) {
        $mfa_data = getMFAData($method->auth_type);
        $mfa_method_labels[] = $mfa_data['label'];
        $mfa_method_counts[] = (int)$method->method_count;
        $mfa_method_colors[] = $mfa_data['color'];
    }

    if (empty($mfa_method_labels)) {
        $mfa_method_labels = ['None'];
        $mfa_method_counts = [1];
        $mfa_method_colors = ['#ccc'];
    }

    $summary_data = [
        'total_users' => $total_users_count,
        'enabled_users' => $enabled_users,
        'not_enabled_users' => $not_enabled_users,
        'mfa_logins' => $mfa_logins,
        'mfa_success_rate' => $mfa_success_rate,
        'locked_accounts' => $locked_accounts,
        'unlocked_accounts' => $unlocked_accounts,
        'mfa_method_labels' => $mfa_method_labels,
        'mfa_method_counts' => $mfa_method_counts,
        'mfa_method_colors' => $mfa_method_colors
    ];

    echo '<script type="text/javascript">
        var mfaData = ' . json_encode($summary_data) . ';
    </script>';
}

?>

<div class="container-fluid py-4">
    <!-- Summary Dashboard -->
    <h5 class="text-center mb-4" style="font-weight:700;font-size: 25px;text-align: center;">Summary Dashboard</h5>
    <div class="row mb-5">
        <div class="col-md-3">
            <canvas id="enabledUsersChart"></canvas>
            <div class="metric-label">Total Users Enabled for MFA</div>
        </div>
        <div class="col-md-3">
            <canvas id="mfaMethodsChart"></canvas>
            <div class="metric-label">MFA Methods</div>
        </div>
        <div class="col-md-3">
            <canvas id="mfaSuccessRateChart"></canvas>
            <div class="metric-label">MFA Success Rate</div>
        </div>
        <div class="col-md-3">
            <canvas id="lockedAccountsChart"></canvas>
            <div class="metric-label">Locked Accounts Due to Grace Period Expiry</div>
        </div>
        <!--<div class="col-md-3 mt-4">
            <canvas id="mfaLoginsChart"></canvas>
            <div class="metric-label">Total Logins with MFA</div>
        </div> -->
    </div>
</div>

<?php
// Call the function to generate data for charts
get_mfa_summary_data();
?>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

<script type="text/javascript">
window.onload = function() {
    // Fallback if mfaData is missing
    if (!window.mfaData || typeof mfaData.total_users !== 'number') {
        mfaData = {
            total_users: 1,
            enabled_users: 0,
            not_enabled_users: 1,
            mfa_success_rate: 0,
            locked_accounts: 0,
            unlocked_accounts: 1,
            mfa_method_labels: ['None'],
            mfa_method_counts: [1],
            mfa_method_colors: ['#ccc']
        };
    }

    const enabledUsersChart = new Chart(document.getElementById('enabledUsersChart'), {
        type: 'doughnut',
        data: {
            labels: ['Enabled', 'Not Enabled'],
            datasets: [{
                data: [mfaData.enabled_users, mfaData.not_enabled_users],
                backgroundColor: ['#16478c', '#eb8c22'],
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: { position: 'top' },
                tooltip: {
                    callbacks: {
                        label: ctx => `${ctx.raw} users`
                    }
                }
            }
        }
    });

    const mfaSuccessRateChart = new Chart(document.getElementById('mfaSuccessRateChart'), {
        type: 'doughnut',
        data: {
            labels: ['Failed', 'Success'],
            datasets: [{
                data: [100 - mfaData.mfa_success_rate, mfaData.mfa_success_rate],
                backgroundColor: ['#eb8c22', '#16478c'],
                borderWidth: 1
            }]
        },
        options: { responsive: true }
    });

    const lockedAccountsChart = new Chart(document.getElementById('lockedAccountsChart'), {
        type: 'doughnut',
        data: {
            labels: ['Locked', 'Unlocked'],
            datasets: [{
                data: [mfaData.locked_accounts, mfaData.unlocked_accounts],
                backgroundColor: ['#eb8c22', '#16478c'],
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            plugins: {
                tooltip: {
                    callbacks: {
                        label: ctx => {
                            let total = mfaData.locked_accounts + mfaData.unlocked_accounts;
                            let percent = ((ctx.raw / total) * 100).toFixed(1);
                            return `${ctx.label}: ${ctx.raw} users (${percent}%)`;
                        }
                    }
                }
            }
        }
    });

    const mfaMethodsChart = new Chart(document.getElementById('mfaMethodsChart'), {
        type: 'doughnut',
        data: {
            labels: mfaData.mfa_method_labels,
            datasets: [{
                data: mfaData.mfa_method_counts,
                backgroundColor: mfaData.mfa_method_colors,
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: { position: 'top' },
                tooltip: {
                    callbacks: {
                        label: ctx => `${ctx.raw} users`
                    }
                }
            }
        }
    });
};
</script>




<style>
    .message {
    margin-top: 10px;
    font-size: 14px;
    padding: 5px;
}

.message.success {
    color: green;
}

.message.error {
    color: red;
}

.message.processing {
    color: orange;
}

/* Activate button style */
.toggle-status-btn.activate {
    background-color: #28a745;  /* Green */
    color: white;
}

/* Deactivate button style */
.toggle-status-btn.deactivate {
    background-color: #dc3545;  /* Red */
    color: white;
}

/* Common button style */
.toggle-status-btn {
    padding: 5px 10px;
    border: none;
    border-radius: 5px;
    cursor: pointer;
    /* font-size: 14px; */
    margin: 0 5px;
}

/* Add hover effects for both buttons */
.toggle-status-btn:hover {
    opacity: 0.8;
}

.generate-recovery-codes {
    background-color: #16478c; /* Blue */
    color: white;
    padding: 5px 10px;
    border: none;
    border-radius: 5px;
    /* font-size: 14px; */
    cursor: pointer;
}
.reset-mfa-btn {
    background-color: #16478c; /* Blue */
    color: white;
}

.generate-recovery-codes:hover {
    background-color: #16478c; /* Darker blue on hover */
}

</style>

<?php
function get_all_users_with_bca_entries() {
    global $wpdb;

    $users = get_users();

    if (empty($users)) {
        echo '<p>No users found.</p>';
        return;
    }

    echo '<h5 class="mb-4 text-center" style="font-weight:700;font-size:25px;">User Activity Log</h5>';

    // Search + Filter UI
    echo '
    <div class="mb-4 d-flex justify-content-between align-items-center">
        <input type="text" id="searchInput" class="form-control me-3" style="max-width: 300px;" placeholder="Search by username, email, role...">
        <select id="mfaFilter" class="form-select" style="max-width: 200px;">
            <option disabled selected>Filter by MFA Status</option>
            <option value="">All Status</option>
            <option value="Enabled">Enabled</option>
            <option value="Not Enabled">Not Enabled</option>
        </select>
    </div>
    ';

    echo '<div class="table-responsive">';
    echo '<table class="table table-bordered">';
    echo '<thead>
        <tr>
            <th>Username ▼</th>
            <th>Email Address</th>
            <th>Role ▼</th>
            <th>MFA Status ▼</th>
            <th>MFA Method ▼</th>
            <th>Last Login ▼</th>
            <th>Actions</th>
        </tr>
    </thead><tbody>';

    foreach ($users as $user) {
        $user_email = $user->user_email;

        $entry = $wpdb->get_row(
            $wpdb->prepare("
                SELECT qr_status, auth_type, user_status, modified_at
                FROM {$wpdb->prefix}bca_api_entries
                WHERE user_email = %s
                ORDER BY modified_at DESC
                LIMIT 1
            ", $user_email)
        );

        $qr_status = $entry ? ($entry->qr_status === 'scanned' ? 'Enabled' : 'Not Enabled') : 'Not Enabled';
        $auth_type = $entry->auth_type ?? 'Not Enabled';
        $modified_at = $entry->modified_at ?? 'Not Enabled';

        $account_status = get_user_meta($user->ID, 'account_status', true);
        $is_inactive = $account_status === 'inactive';

        echo '<tr>';
        echo '<td>' . esc_html(ucwords($user->user_login)) . '</td>';
        echo '<td>' . esc_html($user_email) . '</td>';
        echo '<td>' . esc_html(ucwords(implode(', ', $user->roles))) . '</td>';
        echo '<td>' . esc_html($qr_status) . '</td>';
        echo '<td>' . esc_html(ucwords($auth_type)) . '</td>';
        echo '<td>' . esc_html($modified_at) . '</td>';
        echo '<td>
                <button class="btn btn-sm btn-warning reset-mfa-btn" data-email="' . esc_attr($user_email) . '">Reset MFA</button>
                <button class="btn btn-sm btn-' . ($is_inactive ? 'success' : 'danger') . ' toggle-status-btn"
                        data-email="' . esc_attr($user_email) . '"
                        data-status="' . ($is_inactive ? 'active' : 'inactive') . '">'
                        . ($is_inactive ? 'Activate' : 'Deactivate') .
                '</button>
                <button class="btn btn-sm btn-primary generate-recovery-codes" data-email="' . esc_attr($user_email) . '">Recovery Code</button>
            </td>';
        echo '</tr>';
    }

    echo '</tbody></table></div>';
}
get_all_users_with_bca_entries();
?>

<script>
jQuery(function($) {
    function showMessage(button, message, type = 'success') {
        button.next('.message').remove();
        button.after(`<div class="message ${type}">${message}</div>`);
    }

    function reloadAfterDelay(delay = 1000) {
        setTimeout(() => location.reload(), delay);
    }

    // Reset MFA
    $('.reset-mfa-btn').on('click', function() {
        const btn = $(this);
        const email = btn.data('email');
        showMessage(btn, 'Processing...', 'info');

        $.post(myPluginAjax.ajax_url, {
            action: 'reset_mfa_data',
            user_email: email,
            reset_mfa_nonce: myPluginAjax.reset_mfa_nonce
        }).done(response => {
            showMessage(btn, response.success ? 'MFA data has been reset.' : response.data.message, response.success ? 'success' : 'error');
            if (response.success) reloadAfterDelay();
        }).fail(() => {
            showMessage(btn, 'AJAX error occurred.', 'error');
        });
    });

    // Toggle Account Status
    $('.toggle-status-btn').on('click', function() {
        const btn = $(this);
        const email = btn.data('email');
        const newStatus = btn.data('status');

        showMessage(btn, 'Processing...', 'info');

        $.post(myPluginAjax.ajax_url, {
            action: 'update_account_status',
            user_email: email,
            account_status: newStatus,
            nonce: myPluginAjax.toggle_status_nonce
        }).done(response => {
            showMessage(btn, response.success ? 'Account status updated.' : response.data.message, response.success ? 'success' : 'error');
            if (response.success) reloadAfterDelay();
        }).fail(() => {
            showMessage(btn, 'Failed to update status.', 'error');
        });
    });

    // Generate Recovery Codes
    $('.generate-recovery-codes').on('click', function() {
        const btn = $(this);
        const email = btn.data('email');
        showMessage(btn, 'Generating recovery codes...', 'info');

        $.post(myPluginAjax.ajax_url, {
            action: 'generate_new_recovery_codes_for_user',
            user_email: email,
            nonce: myPluginAjax.nonce
        }).done(response => {
            if (response.success) {
                $('#recovery-codes-output').html('<p>Use the below recovery codes:</p><ul>' +
                    response.data.codes.map(code => `<li>${code}</li>`).join('') + '</ul>');
                $('#qr-code-container').html(`<img src="${response.data.qr_code_url}" alt="QR Code">`);
                showMessage(btn, 'Recovery codes generated.', 'success');
                reloadAfterDelay();
            } else {
                showMessage(btn, response.data.message, 'error');
            }
        }).fail(() => {
            showMessage(btn, 'Failed to generate recovery codes.', 'error');
        });
    });

    // Search + Filter
    function filterTable() {
        const search = $('#searchInput').val().toLowerCase();
        const filter = $('#mfaFilter').val();

        $('table tbody tr').each(function() {
            const row = $(this);
            const username = row.find('td:nth-child(1)').text().toLowerCase();
            const email = row.find('td:nth-child(2)').text().toLowerCase();
            const role = row.find('td:nth-child(3)').text().toLowerCase();
            const mfaStatus = row.find('td:nth-child(4)').text();

            const matchSearch = username.includes(search) || email.includes(search) || role.includes(search);
            const matchMFA = !filter || mfaStatus === filter;

            row.toggle(matchSearch && matchMFA);
        });
    }

    $('#searchInput, #mfaFilter').on('input change', filterTable);
});
</script>

