<?php
class BCA_Report {
    public static function render_tab() {
        if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bca_report'])) {
            // Verify nonce
            if (!isset($_POST['bca_report_nonce']) || !wp_verify_nonce($_POST['bca_report_nonce'], 'bca_save_report')) {
                echo '<div class="alert alert-danger">Nonce verification failed. Please try again.</div>';
                return;
            }

            // Save MFA
            update_option('bca_report', $mfa);
            echo '<div class="alert alert-success">Email Setting saved successfully.</div>';
        }

        $mfa = get_option('bca_report', []);
        include BCA_PLUGIN_PATH . 'templates/report-tab.php';
    }
}


function reset_mfa_data() {
    global $wpdb;

    // Check nonce for security
    if (!isset($_POST['reset_mfa_nonce']) || !wp_verify_nonce($_POST['reset_mfa_nonce'], 'reset_mfa_nonce')) {
        wp_send_json_error(['message' => 'Invalid nonce']);
        return;
    }

    // Check if the email is provided in the AJAX request
    if (isset($_POST['user_email'])) {
        $user_email = sanitize_email($_POST['user_email']);

        // Proceed with the database update
        $update_result = $wpdb->update(
            "{$wpdb->prefix}bca_api_entries",  // Table name
            [
                'qr_status' => 'not_scanned',  // Reset qr_status to 'scanned'
                'TOTP' => NULL,  // Reset TOTP to an empty string
                'qr_code' => NULL,  // Reset qr_code to an empty string
                'email_otp' => NULL,  // Reset email_otp to an empty string
                'phone_otp' => NULL,  // Reset phone_otp to an empty string
                'auth_type' => NULL,  // Reset auth_type to an empty string
                'verify' => NULL, // Reset
                'totp_config' => NULL,
                'otpEmail_config' => NULL,
                'pushnotification_config' => NULL,
                'securityQuestion_config' => NULL,
                'default_mfa_method' => NULL,
                'user_status' => 'logout', // Reset
            ],
            ['user_email' => $user_email],  // Condition: matching user email
            array('%s', '%d'),              // Format for each value (1 string, 1 integer)
            ['%s']  // Format for the WHERE clause (user_email is the condition)
        );

        // Check if the update was successful
        if ($update_result !== false) {
            wp_send_json_success(['message' => 'MFA data reset successfully.']);
        } else {
            wp_send_json_error(['message' => 'Database update failed', 'details' => $wpdb->last_error]);
        }
    } else {
        wp_send_json_error(['message' => 'No user email provided']);
    }
}

add_action('wp_ajax_reset_mfa_data', 'reset_mfa_data');
add_action('wp_ajax_nopriv_reset_mfa_data', 'reset_mfa_data');  // For non-logged-in users if needed



// Handle account status update via AJAX
// function handle_toggle_account_status() {
//     if (isset($_POST['nonce']) && wp_verify_nonce($_POST['nonce'], 'toggle_account_status_nonce')) {
//         $user_email = sanitize_email($_POST['user_email']);
//         $new_status = sanitize_text_field($_POST['account_status']);  // 'active' or 'inactive'

//         // Validate account status value
//         if (!in_array($new_status, ['active', 'inactive'])) {
//             wp_send_json_error(['message' => 'Invalid account status.']);
//         }

//         // Get user by email
//         $user = get_user_by('email', $user_email);

//         if (!$user) {
//             wp_send_json_error(['message' => 'User not found.']);
//         }

//         // Update the account status user meta
//         $today = current_time('mysql');
//         update_user_meta($user->ID, '_setup_later_date', $today);
//         update_user_meta($user->ID, 'account_status', $new_status);

//         wp_send_json_success(['message' => 'Account status updated successfully.']);
//     } else {
//         wp_send_json_error(['message' => 'Nonce verification failed.']);
//     }
// }
// add_action('wp_ajax_update_account_status', 'handle_toggle_account_status');

function handle_toggle_account_status() {
    if (isset($_POST['nonce']) && wp_verify_nonce($_POST['nonce'], 'toggle_account_status_nonce')) {
        $user_email = sanitize_email($_POST['user_email']);
        $new_status = sanitize_text_field($_POST['account_status']);  // 'active' or 'inactive'

        // Validate account status value
        if (!in_array($new_status, ['active', 'inactive'])) {
            wp_send_json_error(['message' => 'Invalid account status.']);
        }

        // Get user by email
        $user = get_user_by('email', $user_email);

        if (!$user) {
            wp_send_json_error(['message' => 'User not found.']);
        }

        // Update the account status user meta
        $today = current_time('mysql');
        update_user_meta($user->ID, '_setup_later_date', $today);
        update_user_meta($user->ID, 'account_status', $new_status);

        // Get the email template for the status change
        global $wpdb;
        $template_name = ($new_status == 'active') ? 'Account Unlocked – Action Required' : 'Account Locked Notification';
        $template = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}bca_email_templates WHERE template_name = %s LIMIT 1",
                $template_name
            )
        );

        if ($template) {
            // Retrieve values from the template
            $subject = $template->email_subject;
            $message = $template->email_message;
            $from_email = $template->from_email;
            $from_name = $template->from_name;

            // Replace placeholders in the subject and message body
            $subject = replace_email_placeholders($subject, $user, $new_status);
            $message = replace_email_placeholders($message, $user, $new_status);

            // Set custom email headers
            $headers = array(
                'Content-Type: text/html; charset=UTF-8',
                'From: ' . $from_name . ' <' . $from_email . '>',
                'Reply-To: ' . $from_email,
            );

            // Send the email
            $mail_sent = wp_mail($user_email, $subject, $message, $headers);

            if ($mail_sent) {
                wp_send_json_success(array('message' => 'Account status updated successfully, and email sent.'));
            } else {
                wp_send_json_error(array('message' => 'Account status updated, but email failed to send.'));
            }
        } else {
            wp_send_json_error(array('message' => 'No active email template found for the status change.'));
        }
    } else {
        wp_send_json_error(['message' => 'Nonce verification failed.']);
    }
}
add_action('wp_ajax_update_account_status', 'handle_toggle_account_status');


// Generate recovery code 

function generate_new_recovery_codes_for_user_by_email($user_email) {
    global $wpdb;

    // Check if the user email is valid
    if (empty($user_email)) {
        return ['success' => false, 'message' => 'Invalid user email.'];
    }

    // Generate 5 unique 6-digit codes
    $codes = [];
    for ($i = 0; $i < 5; $i++) {
        $codes[] = str_pad(rand(100000, 999999), 6, '0', STR_PAD_LEFT); // 6-digit code
    }

    // The table name for recovery codes
    $table_name = $wpdb->prefix . 'bca_recovery_code';

    // Check if there's an existing entry for this user email
    $existing_entry = $wpdb->get_row(
        $wpdb->prepare("SELECT * FROM $table_name WHERE user_email = %s", $user_email)
    );

    if ($existing_entry) {
        // If an existing record is found, update it with new codes
        $result = $wpdb->update(
            $table_name,
            [
                'code_1' => $codes[0],
                'code_2' => $codes[1],
                'code_3' => $codes[2],
                'code_4' => $codes[3],
                'code_5' => $codes[4],
                'status_1' => false,
                'status_2' => false,
                'status_3' => false,
                'status_4' => false,
                'status_5' => false,
                'recovery_completed' => false
            ],
            ['user_email' => $user_email]
        );

        if ($result === false) {
            return ['success' => false, 'message' => 'Error updating recovery codes.'];
        }

        // Successfully updated
        return [
            'success' => true,
            'codes' => $codes,
            'message' => 'Successfully updated recovery codes.'
        ];
    } else {
        // If no existing record, insert new recovery codes
        $result = $wpdb->insert(
            $table_name,
            [
                'user_email' => $user_email,
                'code_1' => $codes[0],
                'code_2' => $codes[1],
                'code_3' => $codes[2],
                'code_4' => $codes[3],
                'code_5' => $codes[4],
                'status_1' => false,
                'status_2' => false,
                'status_3' => false,
                'status_4' => false,
                'status_5' => false,
                'recovery_completed' => false
            ]
        );

        if ($result === false) {
            return ['success' => false, 'message' => 'Error inserting recovery codes.'];
        }

        // Successfully inserted
        return [
            'success' => true,
            'codes' => $codes,
            'message' => 'Successfully generated new recovery codes.'
        ];
    }
}


function handle_generate_new_recovery_codes_for_user_ajax() {
    // Verify the nonce for security
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'generate_recovery_codes_nonce')) {
        wp_send_json_error(['message' => 'Invalid nonce']);
    }

    // Ensure the user email is provided in the request
    if (!isset($_POST['user_email']) || empty($_POST['user_email'])) {
        wp_send_json_error(['message' => 'User email is required']);
    }

    $user_email = sanitize_email($_POST['user_email']); // Sanitize the email input

    // Call the function to generate recovery codes for the specified user
    $result = generate_new_recovery_codes_for_user_by_email($user_email);

    // Return the response
    if ($result['success']) {
        wp_send_json_success($result);
    } else {
        wp_send_json_error(['message' => $result['message']]);
    }
}

add_action('wp_ajax_generate_new_recovery_codes_for_user', 'handle_generate_new_recovery_codes_for_user_ajax');
