<?php
function bca_create_tenant( $company_name, $company_email ) {
    global $wpdb;

    // Step 1: Check if the tenant already exists in the bca_master table
    $existing_tenant = $wpdb->get_row(
        $wpdb->prepare( "SELECT tenant_name, username, api_key FROM {$wpdb->prefix}bca_master WHERE company_email = %s", $company_email )
    );

    // Step 2: If the tenant exists, just update the bca_api_entries table
    if ( $existing_tenant ) {
        // Extract tenant data from existing record
        $tenant_name = $existing_tenant->tenant_name;
        $username = $existing_tenant->username;

        // Insert into bca_api_entries table (new entry for the same tenant)
        $api_entries_table = $wpdb->prefix . 'bca_api_entries';
        $wpdb->insert(
            $api_entries_table,
            [
                'tenant_name' => $tenant_name,
                'user_email' => $company_email,
                'username' => $username,
                'external_source' => true,
            ]
        );

        // Insert an entry for the logged-in user if different from admin
        $current_user = wp_get_current_user();
        $admin_email = get_option( 'admin_email' );

        // Only add the logged-in user entry if they are different from the site admin
        if ( $current_user->user_email !== $admin_email ) {
            $wpdb->insert(
                $api_entries_table,
                [
                    'tenant_name' => $tenant_name,
                    'user_email' => $current_user->user_email,
                    'username' => $current_user->user_login,
                    'external_source' => false, // Logged-in user, not external source
                ]
            );
        }

        // Return the ID of the existing tenant from bca_master
        return $wpdb->get_var(
            $wpdb->prepare( "SELECT id FROM {$wpdb->prefix}bca_master WHERE company_email = %s", $company_email )
        );
    }

    // Step 3: If the tenant doesn't exist, create it via the API
    $api_url = 'https://pushnotifications.baariga.com/api/v1/tenant/create';
    $data = [
        "company_name" => strtolower( $company_name ), 
        "company_email" => $company_email,
        "external_source" => true, 
    ];

    // Send the API request
    $response = bca_send_api_request( $api_url, $data );

    if ( is_wp_error( $response ) ) {
        return $response; 
    }

    // Decode the API response
    $response_data = json_decode( $response, true );

    if ( isset( $response_data['tenantName'] ) && isset( $response_data['serviceUsersInfo'][0] ) ) {
        // Extract tenant info from API response
        $tenant_name = $response_data['tenantName'];
        $username = $response_data['serviceUsersInfo'][0]['username'];
        $api_key = $response_data['serviceUsersInfo'][0]['apikey'];

        // Insert into bca_master table (new tenant)
        $master_table = $wpdb->prefix . 'bca_master';
        $wpdb->insert(
            $master_table,
            [
                'tenant_name' => $tenant_name,
                'company_email' => $company_email,
                'api_key' => $api_key,
                'external_source' => true,
                'username' => $username,
            ]
        );

        // Insert into bca_api_entries table for this new tenant
        $api_entries_table = $wpdb->prefix . 'bca_api_entries';
        $wpdb->insert(
            $api_entries_table,
            [
                'tenant_name' => $tenant_name,
                'user_email' => $company_email,
                'username' => $username,
                'external_source' => true,
            ]
        );

        // Insert an entry for the logged-in user if different from admin
        $current_user = wp_get_current_user();
        $admin_email = get_option( 'admin_email' );

        // Only add the logged-in user entry if they are different from the site admin
        if ( $current_user->user_email !== $admin_email ) {
            $wpdb->insert(
                $api_entries_table,
                [
                    'tenant_name' => $tenant_name,
                    'user_email' => $current_user->user_email,
                    'username' => $current_user->user_login,
                    'external_source' => false, // Logged-in user, not external source
                ]
            );
        }

        // Return the ID of the newly created tenant from bca_master
        return $wpdb->insert_id;
    }

    // Return error if API creation fails
    return new WP_Error( 'api_error', 'Failed to create tenant via the API.' );
}




function bca_generate_qr_code( $user_email, $tenant, $id ) {
    global $wpdb;

    $qr_api_url = 'https://pushnotifications.baariga.com/api/v1/push-notify/generate-qr-phone';

    $headers = [
        "tanant: $tenant", 
        "ID: $id",         
        'Content-Type: application/json'
    ];
    $data = [ "user_id" => $user_email ];

    $ch = curl_init( $qr_api_url );
    curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
    curl_setopt( $ch, CURLOPT_HTTPHEADER, $headers );
    curl_setopt( $ch, CURLOPT_POST, true );
    curl_setopt( $ch, CURLOPT_POSTFIELDS, json_encode( $data ) );

    $response = curl_exec( $ch );

    if ( curl_errno( $ch ) ) {
        curl_close( $ch );
        return new WP_Error( 'curl_error', curl_error( $ch ) );
    }

    curl_close( $ch );

    $response_data = json_decode( $response, true );

    if ( isset( $response_data['code'] ) ) {

        $table_name = $wpdb->prefix . 'bca_api_entries';
        $wpdb->update(
            $table_name,
            [ 'qr_code' => $response_data['code'], 'qr_status' => 'scanned' ],
            [ 'user_email' => $user_email ]
        );

        return $response_data['code']; 
    }

    return new WP_Error( 'qr_error', 'Failed to generate QR code.' );
}

function bca_send_api_request( $url, $data ) {
    $ch = curl_init( $url );
    curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
    curl_setopt( $ch, CURLOPT_HTTPHEADER, [ 'Content-Type: application/json' ] );
    curl_setopt( $ch, CURLOPT_POST, true );
    curl_setopt( $ch, CURLOPT_POSTFIELDS, json_encode( $data ) );

    $response = curl_exec( $ch );

    if ( curl_errno( $ch ) ) {
        curl_close( $ch );
        return new WP_Error( 'curl_error', curl_error( $ch ) );
    }

    curl_close( $ch );
    return $response;
}
